<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\GoldItem;
use App\Models\Customer;
use Illuminate\Http\Request;

class GoldItemController extends Controller
{
    public function index()
    {
        $golditems = GoldItem::with('customer')->latest()->paginate(10);
        return view('admin.gold_items.index', compact('golditems'));
    }

    public function create()
    {
        $customers = Customer::select('id', 'full_name')->get();
        return view('admin.gold_items.create', compact('customers'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'description' => 'required|string|max:255',
            'quantity' => 'required|integer|min:1',
            'gross_weight' => 'required|numeric|min:0',
            'stone_weight' => 'required|numeric|min:0|lte:gross_weight',
            'purity' => 'required|integer|min:1|max:24',
            'rate_per_gram' => 'required|numeric|min:0',
        ]);

        $net_weight = $request->gross_weight - $request->stone_weight;
        $market_value = $net_weight * $request->rate_per_gram;

        GoldItem::create([
            'customer_id' => $request->customer_id,
            'description' => $request->description,
            'quantity' => $request->quantity,
            'gross_weight' => $request->gross_weight,
            'stone_weight' => $request->stone_weight,
            'net_weight' => $net_weight,
            'purity' => $request->purity,
            'rate_per_gram' => $request->rate_per_gram,
            'market_value' => $market_value,
            'remarks' => $request->remarks,
        ]);

        return redirect()->route('admin.gold_items.index')
            ->with('success', 'Gold item added successfully.');
    }

    public function edit(GoldItem $gold_item)
    {
        $customers = Customer::select('id', 'full_name')->get();
        return view('admin.gold_items.edit', compact('gold_item', 'customers'));
    }

    public function update(Request $request, GoldItem $gold_item)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'description' => 'required|string|max:255',
            'quantity' => 'required|integer|min:1',
            'gross_weight' => 'required|numeric|min:0',
            'stone_weight' => 'required|numeric|min:0|lte:gross_weight',
            'purity' => 'required|integer|min:1|max:24',
            'rate_per_gram' => 'required|numeric|min:0',
        ]);

        $net_weight = $request->gross_weight - $request->stone_weight;
        $market_value = $net_weight * $request->rate_per_gram;

        $gold_item->update([
            'customer_id' => $request->customer_id,
            'description' => $request->description,
            'quantity' => $request->quantity,
            'gross_weight' => $request->gross_weight,
            'stone_weight' => $request->stone_weight,
            'net_weight' => $net_weight,
            'purity' => $request->purity,
            'rate_per_gram' => $request->rate_per_gram,
            'market_value' => $market_value,
            'remarks' => $request->remarks,
        ]);

        return redirect()->route('admin.gold_items.index')
            ->with('success', 'Gold item updated successfully.');
    }

    public function destroy(GoldItem $gold_item)
    {
        $gold_item->delete();

        return redirect()->route('admin.gold_items.index')
            ->with('success', 'Gold item deleted successfully.');
    }
}
